-- vim: ts=2 sw=2 ai et
--
-- Copyright(c) 2008 Johny Mattsson
--      a.k.a.
-- Anyia of HordeYakka (Jubei'Thos)
--
-- Permission to copy and re-use as seen fit.
-- Credit would be nice.
--
-- Massive work on French translation done by Elisyaa - Le Consensus (TGC Server) 12.11.2025

local NAME = "TwinCam"
local ADDON_NAME = "TwinCam_Overlay"
TwinCam = LibStub ("AceAddon-3.0"):NewAddon ("TwinCam", "AceConsole-3.0", "AceEvent-3.0", "AceTimer-3.0")
-- Set a more useful __tostring
local mt = getmetatable (TwinCam) or {}
mt['__tostring'] = function () return NAME end

local options = {
  type = 'group',
  handler = TwinCam,
  childGroups = 'tab',
  args = {
    warnings = {
      type = 'group',
      name = "Warnings",
      order = 1,
      args = {
        twins = {
          type = 'group',
          name = "Eredar Twins",
          desc = "Eredar Twins warning settings",
          order = 3,
          args = {
            conflagDesc = {
              type = 'header',
              order = 0,
              name = "Conflagration",
            },
            conflagSound = {
              type = 'toggle',
              order = 1,
              name = "Play sound",
              desc = "Play a warning sound when a conflagration is being cast at you",
              get = "GetVal",
              set = "SetVal",
            },
            conflagVisible = {
              type = 'toggle',
              order = 2,
              name = "Flash screen",
              desc = "Flash a skull overlay across the screen when a conflagration is being cast at you",
              get = "GetVal",
              set = "SetVal",
            },
            conflagColor = {
              type = 'color',
              hasAlpha = false,
              order = 3,
              name = "Skull color",
              desc = "Color/intensity of the skull overlay",
              get = "GetColor",
              set = "SetColor",
            },
            conflagPreview = {
              type = 'execute',
              order = 4,
              name = "Preview",
              desc = "Preview Conflagration warning",
              func = "OnConflag",
            },
            spacer = {
              type = 'description',
              order = 9,
              name = '\n',
            },
            shadowfuryDesc = {
              type = 'header',
              order = 10,
              name = "Shadowfury stun",
            },
            shadowfuryVisible = {
              type = 'toggle',
              name = "Tint screen",
              desc = "Tint the screen when you are stunned by Shadowfury",
              order = 11,
              get = "GetVal",
              set = "SetVal",
            },
            shadowfuryColor = {
              type = 'color',
              hasAlpha = true,
              name = "Tint color",
              desc = "Color to tint the screen during Shadowfury stun",
              order = 12,
              get = "GetColor",
              set = "SetColor",
            },
            shadowfuryPreview = {
              type = 'execute',
              name = "Preview",
              order = 13,
              desc = "Preview Shadowfury stun warning",
              func = "OnShadowfury",
            },
            shadowfuryPreviewOff = {
              type = 'execute',
              name = "Preview Off",
              order = 14,
              desc = "Stop previewing the Shadowfury stun warning",
              func = "StopFlashShadowfury",
            }
          }
        },

        brutallus = {
          type = 'group',
          name = "Brutallus",
          desc = "Brutallus warning settings",
          order = 1,
          args = {
            brutBurnDesc = {
              type = 'header',
              order = 0,
              name = "Burn!",
            },
            brutBurnSound = {
              type = 'toggle',
              order = 1,
              name = "Play sound",
              desc = "Play a warning sound when you get Burn",
              get = "GetVal",
              set = "SetVal",
            },
            brutBurnVisible = {
              type = 'toggle',
              order = 2,
              name = "Flash screen",
              desc = "Flash a warning across the screen when you get Burn",
              get = "GetVal",
              set = "SetVal",
            },
            brutBurnType = {
              type = 'select',
              style = 'dropdown',
              order = 3,
              name = "Warning type",
              desc = "Type of warning to flash on Burn, if enabled",
              get = "GetVal",
              set = "SetVal",
              values = { ["explicit"] = "Explicit", ["tame"] = "Plain" }
            },
            brutBurnPreview = {
              type = 'execute',
              order = 4,
              name = "Preview",
              desc = "Preview Burn warning",
              func = "OnBrutBurn",
            },
          }
        },

        felmyst = {
          type = 'group',
          name = "Felmyst",
          desc = "Failmyst warning settings",
          order = 2,
          args = {
            felmystFrequency = {
              type = 'range',
              order = 3,
              name = "Scan frequency",
              desc = "Frequency to scan for Encapsulate (HZ), higher is better but more resource intensive",
              min = 1, max = 10,
              step = 1, bigStep = 1,
              get = "GetVal",
              set = "SetVal",
            },
            spacer2 = { type = 'description', order = 4, name = '' },
            felmystEncapsDescription = {
              type = 'header',
              order = 5,
              name = "Encapsulate (self)",
            },
            felmystEncapSelfSound = {
              type = 'toggle',
              order = 6,
              name = "Play sound",
              desc = "Play a warning sound when you get targeted for Encapsulate",
              get = "GetVal",
              set = "SetVal",
            },
            felmystEncapSelfVisible = {
              type = 'toggle',
              order = 7,
              name = "Flash screen",
              desc = "Flash a warning across the screen when you get targeted for Encapsulate",
              get = "GetVal",
              set = "SetVal",
            },
            felmystEncapSelfPreview = {
              type = 'execute',
              order = 8,
              name = "Preview",
              desc = "Preview Encapsulate (self) warning",
              func = "felmystEncapSelfTest"
            },
            spacer = { type = 'description', order = 9, name = '\n'
            },
            felmystEncapNearbyDesc = {
              type = 'header',
              order = 10,
              name = "Encapsulate (nearby)"
            },
            felmystEncapNearbySound = {
              type = 'toggle',
              order = 11,
              name = "Play sound",
              desc = "Play a warning sound when a player close to you gets Encapsulated",
              get = "GetVal",
              set = "SetVal",
            },
            felmystEncapNearbyVisible = {
              type = 'toggle',
              order = 12,
              name = "Flash screen",
              desc = "Flash a warning on the screen when a player close to you gets Encapsulated",
              get = "GetVal",
              set = "SetVal",
            },
            felmystEncapNearbyPreview = {
              type = 'execute',
              order = 13,
              name = "Preview",
              desc = "Preview Encapsulate (nearby) warning",
              func = "felmystEncapNearbyTest"
            }
          }
        }
      }
    },

    about = {
      type = 'group',
      name = "About",
      desc = "About " .. ADDON_NAME,
      order = -1,
      args = {
        __ = {
          type = 'header',
          name = ADDON_NAME .. " " .. GetAddOnMetadata (ADDON_NAME, "Version"),
          order = 1,
        },
        _1 = {
          type = 'description',
          name = GetAddOnMetadata (ADDON_NAME, "Notes"),
          order = 10,
        },
        _2 = {
          type = 'description',
          name = "By " .. GetAddOnMetadata(ADDON_NAME, "Author"),
          order = 20,
        },
        _3 = { type = 'description', name = "", order = 30, },
        _4 = {
          type = 'description',
          name = "Copyright(C) 2008 Johny Mattsson",
          order = 40,
        },
        _5 = { type = 'description', name = "", order = 50, },
        _6 = {
          type = 'description',
          name = "Build: " .. GetAddOnMetadata (ADDON_NAME, "X-Build"):match("%d+"),
          order = 60,
        }
      }
    }
  }
}
local defaults = {
  profile = {
    conflagSound = true,
    conflagVisible = true,
    conflagColor = { r = 0.6, g = 0.6, b = 0.6, a = 1 },
    shadowfuryVisible = true,
    shadowfuryColor = { r = 1, g = 0, b = 0, a = 1 },
    brutBurnSound = true,
    brutBurnVisible = true,
    brutBurnType = "explicit",
    felmystEncapSelfSound = true,
    felmystEncapSelfVisible = true,
    felmystEncapNearbySound = true,
    felmystEncapNearbyVisible = true,
    felmystFrequency = 5,
  }
}

-- "Constants"
local GWA = "Grande démoniste Alythess"
local LS = "Dame Sacrocingle"
local FELMYST = "Gangrebrume"
local RunAwaySound = "Sound/Creature/HoodWolf/HoodWolfTransformPlayer01.wav"
local LamentSound = "Sound/Music/GlueScreenMusic/BCCredits_Lament_of_the_Highborne.mp3"
local LightYouUpSound = "Sound/Creature/MillhouseManastorm/TEMPEST_Millhouse_Pyro01.wav"
local AbandonSound = "Sound/Creature/CThun/CThunYourFriends.wav"
-- We actually snarf the texture straight from the TargetRaidTargetIcon frame,
-- but in case we ever need to remember where it is, this is it
local RaidTargetsTextures = "Interface/TargetingFrame/UI-RaidTargetingIcons"
local TexturesPath = "Interface\\AddOns\\" .. ADDON_NAME .. "\\Textures"
local BurnExplicit = TexturesPath .. "\\burnexplicit"
local BurnTame = TexturesPath .. "\\burnplain"
local BurnYHBFFFFF = TexturesPath .. "\\YHBFFFFF"
local BurnLIGYBA = TexturesPath .. "\\LIGYBA"

-- Config manipulators
function TwinCam:GetVal (info)
  return self.db.profile[info[#info]]
end
function TwinCam:SetVal (info, v)
  self.db.profile[info[#info]] = v
  if info[#info] == "brutBurnType" then self:PickBurnWarning () end
end
function TwinCam:GetColor (info)
  local c = self.db.profile[info[#info]]
  return c.r, c.g, c.b, (c.a or 1)
end
function TwinCam:SetColor (info, r, g, b, a)
  local c = { r = r, g = g, b = b, a = a }
  self.db.profile[info[#info]] = c
end

-- Helpers for debugging
function TwinCam:debug_sf ()
  TwinCam:COMBAT_LOG_EVENT_UNFILTERED ("", 0, "SPELL_AURA_APPLIED",0x0000000000000000,nil,0x80000000,0x000000000194E5F2,UnitName("player"),0x514,45270,"Furie de l'ombre",0x20,"DEBUFF")-- Translation from en
end

function TwinCam:debug_emote()
  TwinCam:CHAT_MSG_RAID_BOSS_EMOTE("CHAT_MSG_RAID_BOSS_EMOTE", "Alythess lance une Déflagration vers quelqu'un", GWA, "fr", "whatever", self.name)-- Translation from en "Alythess directs Conflagration at Someone"
end

function TwinCam:debug_burn ()
  TwinCam:COMBAT_LOG_EVENT_UNFILTERED ("", 0, "SPELL_AURA_APPLIED",0x0000000000000000,nil,0x80000000,0x000000000195195D,UnitName("player"),0x512,46394,"Brûler",0x4,"DEBUFF")-- Translation from en
end

function TwinCam:debug_felmystp1 ()
  TwinCam:COMBAT_LOG_EVENT_UNFILTERED ("", 0, "SPELL_CAST_SUCCESS",0xF1300061CE00033E,FELMYST,0x10a48,0x0000000000000000,nil,0x80000000,47002,"Vapeurs nauséabondes",0x8)-- Translation from en
end
function TwinCam:debug_felmystp2 ()
  TwinCam:CHAT_MSG_MONSTER_YELL ("", "Je suis plus forte que jamais !", FELMYST, "lang", "", nil)-- Translation from en "I am stronger than ever!"
end


-- Detect Conflagration via the boss emote
function TwinCam:CHAT_MSG_RAID_BOSS_EMOTE(eventName, msg, sender, lang, channel, target)
  if ((sender == GWA) or (sender == LS)) and (target == self.name) and
     (msg:match (".*Déflagration.*"))-- Translation from Conflagration
  then
    self:OnConflag ();
  end
end

-- Felmyst's airphase announcement
function TwinCam:CHAT_MSG_MONSTER_YELL(eventName, msg, sender, lang, channel, target)
  if (sender == FELMYST) and (msg:match (".*forte que jamais.*")) then -- Translation from stronger than ever
     self:StopFelmystP1 ()
     -- P1 restarts in 100 secs, re-use existing Felmyst id
     if (self.felmyst.debug) then
       self:Print ("twincam debug: scheduling p1 in 100 sec")
     end
     self.felmyst.p1timer = self:ScheduleTimer ("StartFelmystP1", 100, nil)
  end
end

function TwinCam:COMBAT_LOG_EVENT_UNFILTERED(eventName, time, event, srcId, srcName, srcFlags, dstID, dstName, dstFlags, spellID, spellName)
  -- Only care about shadow fury affecting the player
  if (spellName == "Furie de l'ombre") and (dstName == self.name) then --Translation from Shadowfury
    if (event == "SPELL_AURA_APPLIED") then
      self:OnShadowfury ()
    elseif (event == "SPELL_AURA_REMOVED") then
      self:StopFlashShadowfury ()
    end
  end
  -- Burn on player
  if (spellName == "Brûler") and (dstName == self.name) then --Transaltion from Burn
    if (event == "SPELL_AURA_APPLIED") then
      self:OnBrutBurn ()
    end
  end

  -- The cast of Noxious Fumes kicks off the Felmyst encounter for us
  if (srcName == FELMYST) and (spellName == "Vapeurs nauséabondes") and -- Translation from Noxious Fumes
     (event == "SPELL_CAST_SUCCESS") then self:StartFelmystP1() end
  -- If Felmyst dies, shut down the target scanning
  if (event == "UNIT_DIED") and (dstName == FELMYST) then
    self:StopFelmystP1 () -- if we're in p1
    self:CancelTimer (self.felmyst.p1timer, true) -- if we're in p2
  end
end

function TwinCam:PLAYER_DEAD(eventName)
  self:StopFlashShadowfury ()
  --We don't explicitly stop any Felmyst timers,but let them expire on their own
end

function TwinCam:OnConflag ()
  if self.db.profile.conflagSound then PlaySoundFile (RunAwaySound) end
  if self.db.profile.conflagVisible then
    local c = self.db.profile.conflagColor
    self.conflagtex:SetVertexColor (c.r, c.g, c.b)
    UIFrameFlash (self.conflagtex, 0, 0.5, 5.1, false, 0, 0)
  end
end

function TwinCam:OnShadowfury ()
  if self.db.profile.shadowfuryVisible then
    local c = self.db.profile.shadowfuryColor
    self.sftex:SetVertexColor (c.r, c.g, c.b)
    self.sftex:SetAlpha (c.a)
  end
end

function TwinCam:StopFlashShadowfury ()
  self.sftex:SetAlpha (0)
end

function TwinCam:OnBrutBurn ()
  if self.db.profile.brutBurnVisible then
    UIFrameFlash (self.burntex, 0.5, 3, 3.55, false, 0, 0)
  end
  if self.db.profile.brutBurnSound then PlaySoundFile (LightYouUpSound) end
end


function TwinCam:felmystEncapSelfTest () self:OnFelmystEncaps (self.name) end
function TwinCam:felmystEncapNearbyTest () self:OnFelmystEncaps ("-TEST-") end
function TwinCam:OnFelmystEncaps (target)
  if target == self.name then
    if self.db.profile.felmystEncapSelfSound then
      PlaySoundFile (AbandonSound)
    end
    if self.db.profile.felmystEncapSelfVisible then
      UIFrameFlash (self.encapselftex, 0, 0.6, 0.6, false, 0, 0)
    end
  else
    -- 1 Inspect 28yd, 2 Trade 11.11 yd, 3 Duel 9.9yd, 4 Follow 28yd
    local inRange = CheckInteractDistance (target, 2) 
    if (target == "-TEST-") then inRange = 1 end -- local test
    if inRange then
      if self.db.profile.felmystEncapNearbySound then
        PlaySoundFile (RunAwaySound)
      end
      if self.db.profile.felmystEncapNearbyVisible then
        self.encapnearbytex:SetVertexColor (0.8, 0, 0)
        UIFrameFlash (self.encapnearbytex, 0, 0.2, 0.2, false, 0, 0)
      end
    end
  end
end

function TwinCam:StartFelmystP1 ()
  if self.felmyst.debug then self:Print("twincam debug: start Felmyst P1") end
  self.felmyst.checktgt = true
  self.felmyst.last = nil
  self.felmyst.scantimer = self:ScheduleRepeatingTimer ("InFelmystP1",
    1 / self.db.profile.felmystFrequency, nil)
  -- make sure we don't get stuck in p1 forever, 65 secs should be plenty...
  self.felmyst.proceedtop2timer = self:ScheduleTimer ("StopFelmystP1", 65, nil)
end

function TwinCam:InFelmystP1 ()
  local id = self:FindTarget (FELMYST)
  if not id and self.felmyst.debug then
    self:Print ("twincam debug: no target for Felmyst found")
  end

  if self.felmyst.checktgt and id then
    local tgt = UnitName (id .. "target")
    if self.felmyst.debug then
      self:Print("twincam debug: felmyst target is " .. (tgt or "nil"))
    end
    if tgt and tgt ~= self.felmyst.last then
      if not GetPartyAssignment ("maintank", tgt) then
        -- switched target to a non-tank, awooga!
        self:OnFelmystEncaps (tgt)
      end
    end
    self.felmyst.last = tgt
  end
end

function TwinCam:StopFelmystP1 ()
  if self.felmyst.debug then self:Print ("twincam debug: end Felmyst P1") end
  self.felmyst.checktgt = nil
  self:CancelTimer (self.felmyst.scantimer, true)
end

function TwinCam:FindTarget (tgt, lastknown)
  if lastknown and UnitName (lastknown) == tgt then return lastknown end
  if UnitName ("target") == tgt then return "target" end
  if UnitName ("focus") == tgt then return "focus" end
  if UnitName ("focustarget") == tgt then return "focustarget" end

  local count = GetNumRaidMembers ()
  local format = "raid%dtarget"
  if (count == 0) then
    count = GetNumPartyMembers ()
    format = "party%dtarget"
  end
  if count == 0 then return nil end
  for i = 1, count do
    local chktgt = string.format (format, i)
    if UnitName(chktgt) == tgt then return chktgt end
  end
  return nil
end

function TwinCam:PickBurnWarning ()
  if self.burntex then
    UIFrameFlashRemoveFrame (self.burntex)
    self.burntex:SetAlpha (0)
  end
  local burntex = self.overlayFrame:CreateTexture ("BurnTexture")
  burntex:SetAlpha (0)
  burntex:SetBlendMode ("ADD")
  burntex:SetAllPoints (overlayframe)
  local burntype = self.db.profile.brutBurnType
  if burntype == "tame" then
    burntex:SetTexture (BurnTame)
  elseif burntype == "explicit" then
    burntex:SetTexture (BurnExplicit)
  elseif burntype == "yhbfffff" then
    burntex:SetTexture (BurnYHBFFFFF)
  elseif burntype == "ligyba" then
    burntex:SetTexture (BurnLIGYBA)
  end
  self.burntex = burntex
end

function TwinCam:OnProfileChanged ()
  self:PickBurnWarning ()
end

function TwinCam:OnInitialize ()
  self.name = UnitName ("player")

  -- Felmyst status vars
  self.felmyst = { }

  -- Set up configuration panel
  LibStub ("AceConfigRegistry-3.0"):RegisterOptionsTable (NAME, options)
  local ACD = LibStub ("AceConfigDialog-3.0")
  ACD:AddToBlizOptions (NAME, GetAddOnMetadata (ADDON_NAME, "Title"))
  self:RegisterChatCommand ("twincam", function()
    InterfaceOptionsFrame_OpenToFrame(ACD.BlizOptions[NAME].frame)
  end)
  self:RegisterChatCommand ("twincambonus", function() PlaySoundFile(LamentSound) end)

  -- Prepare config db
  self.db = LibStub ("AceDB-3.0"):New ("TwinCamDB", defaults)
  self.db.RegisterCallback(self, "OnProfileChanged", "OnProfileChanged")
	self.db.RegisterCallback(self, "OnProfileCopied", "OnProfileChanged")
	self.db.RegisterCallback(self, "OnProfileReset", "OnProfileChanged") 
  options.args.profiles = LibStub ("AceDBOptions-3.0"):GetOptionsTable (self.db)

  -- Our overlay frame
  local overlayframe = CreateFrame ("Frame", nil, WorldFrame)
  overlayframe:SetClampedToScreen (true)
  overlayframe:SetFrameStrata ("BACKGROUND")
  overlayframe:SetAllPoints (UIParent)
  overlayframe:Show ()
  self.overlayFrame = overlayframe

  -- Allow for HY optional madness :)
  local probetex = overlayframe:CreateTexture ("TwinCamProbeTexture")
  local burnvals = options.args.warnings.args.brutallus.args.brutBurnType.values
  probetex:SetTexture (nil) probetex:SetTexture (BurnYHBFFFFF)
  if probetex:GetTexture () then 
    defaults.profile.brutBurnType = "yhbfffff";
    burnvals["yhbfffff"] = "Beefo";
  end
  probetex:SetTexture (nil) probetex:SetTexture (BurnLIGYBA)
  if probetex:GetTexture () then
    defaults.profile.brutBurnType = "ligyba";
    burnvals["ligyba"] = "Pewn";
  end
  probetex:SetTexture (nil)

  -- Conflag 'skull' texture (from the raid marker)
  local conflagtex = overlayframe:CreateTexture ("TwinCamConflagTexture")
  conflagtex:SetAlpha (0)
  conflagtex:SetBlendMode ("ADD")
  conflagtex:SetAllPoints (overlayframe)
  conflagtex:SetTexture (TargetRaidTargetIcon:GetTexture ())
  -- 1 star, 2 circle, 3 diamond, 4 triangle, 5 moon, 6 square, 7 cross, 8 skull
  SetRaidTargetIconTexture (conflagtex, 8)
  self.conflagtex = conflagtex

  -- Shadowfury texture (from tooltip background)
  local sftex = overlayframe:CreateTexture ("TwinCamShadowfuryTexture")
  sftex:SetAlpha (0)
  sftex:SetBlendMode ("DISABLE")
  sftex:SetAllPoints (overlayframe)
  sftex:SetTexture ("Interface/Tooltips/UI-Tooltip-Background")
  self.sftex = sftex

  -- Burn texture
  self:PickBurnWarning ()

  -- Encap self texture
  self.encapselftex = self.conflagtex

  -- Encap nearby texture (from tooltip background)
  local encapnearbytex = overlayframe:CreateTexture ("TwinCamShadowfuryTexture")
  encapnearbytex:SetAlpha (0)
  encapnearbytex:SetBlendMode ("DISABLE")
  encapnearbytex:SetAllPoints (overlayframe)
  encapnearbytex:SetTexture ("Interface/Tooltips/UI-Tooltip-Background")
  self.encapnearbytex = encapnearbytex
end

function TwinCam:OnEnable ()
  self:RegisterEvent("COMBAT_LOG_EVENT_UNFILTERED")
  self:RegisterEvent("CHAT_MSG_RAID_BOSS_EMOTE")
  self:RegisterEvent("CHAT_MSG_MONSTER_YELL")
  self:RegisterEvent("PLAYER_DEAD")
  self:Print ("This AddOn may contain explicit language and is recommended for a Mature audience only.")
end

